<?php

if (!defined('ABSPATH')) {
    die;
}

/**
 * WC Dependency Checker
 *
 * Checks if have any dependency
 */
class Foxpost_Wc_Shipping_Dependencies {

	/** @var Foxpost_Wc_Shipping_Dependencies single instance of this class */
	private static $instance;
    
	/** @var array the admin notices to add */
	private $notices = array();
    
	private static $active_plugins;

	/** the plugin name, for displaying notices */
	const PLUGIN_NAME = 'FOXPOST - Packeta Group';
    
	/**
	 * Constructs the class.
	 *
	 * @since 1.0.0
	 */
	protected function __construct() {
        add_action( 'admin_init', array( $this, 'add_plugin_notices' ) );
        add_action( 'admin_notices', array( $this, 'admin_notices' ), 15 );
	}

	public static function init() {
		self::$active_plugins = (array) get_option( 'active_plugins', array() ); 
		if ( is_multisite() )
			self::$active_plugins = array_merge( self::$active_plugins, get_site_option( 'active_sitewide_plugins', array() ) );
	}

	public static function woocommerce_active_check() { 
		if ( ! self::$active_plugins ) self::init();
		return in_array( 'woocommerce/woocommerce.php', self::$active_plugins ) || array_key_exists( 'woocommerce/woocommerce.php', self::$active_plugins );
	}
    
    
    
	/**
	 * Adds notices for out-of-date WordPress and/or WooCommerce versions.
	 *
	 * @since 1.0.0
	 */
	public function add_plugin_notices() {

		if ( ! $this->is_wp_compatible() ) {

			$this->add_admin_notice( 'update_wordpress', 'error', sprintf(
				'%s requires WordPress version %s or higher. Please %supdate WordPress &raquo;%s',
				'<strong>"' . self::PLUGIN_NAME . '"</strong>',
				FOXPOST_WC_SHIPPING_MIN_WP_VERSION,
				'<a href="' . esc_url( admin_url( 'update-core.php' ) ) . '">', '</a>'
			) );
		}
        
        //only test WC version, when is in active state
        if ( ! self::woocommerce_active_check() ) return ;

		if ( ! $this->is_wc_compatible() ) {

			$this->add_admin_notice( 'update_woocommerce', 'error', sprintf(
				'%1$s requires WooCommerce version %2$s or higher. Please %3$supdate WooCommerce%4$s to the latest version, or %5$sdownload the minimum required version &raquo;%6$s',
				'<strong>"' . self::PLUGIN_NAME . '"</strong>',
				FOXPOST_WC_SHIPPING_MIN_WOO_VERSION,
				'<a href="' . esc_url( admin_url( 'update-core.php' ) ) . '">', '</a>',
				'<a href="' . esc_url( 'https://downloads.wordpress.org/plugin/woocommerce.' . FOXPOST_WC_SHIPPING_MIN_WOO_VERSION . '.zip' ) . '">', '</a>'
			) );
		}
	}  

	/**
	 * Determines if the WordPress compatible.
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	protected function is_wp_compatible() {

		if ( ! FOXPOST_WC_SHIPPING_MIN_WP_VERSION ) {
			return true;
		}

		return version_compare( get_bloginfo( 'version' ), FOXPOST_WC_SHIPPING_MIN_WP_VERSION, '>=' );
	}


	/**
	 * Determines if the WooCommerce compatible.
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	protected function is_wc_compatible() {

		if ( ! FOXPOST_WC_SHIPPING_MIN_WOO_VERSION ) {
			return true;
		}

		return defined( 'WC_VERSION' ) && version_compare( WC_VERSION, FOXPOST_WC_SHIPPING_MIN_WOO_VERSION, '>=' );
	}
    
    
	/**
	 * Adds an admin notice to be displayed.
	 *
	 * @since 1.0.0
	 *
	 * @param string $slug the slug for the notice
	 * @param string $class the css class for the notice
	 * @param string $message the notice message
	 */
	public function add_admin_notice( $slug, $class, $message ) {

		$this->notices[ $slug ] = array(
			'class'   => $class,
			'message' => $message
		);
	}


	/**
	 * Displays any admin notices added with Foxpost_Wc_Shipping_Dependencies::add_admin_notice()
	 *
	 * @since 1.0.0
	 */
	public function admin_notices() {

		foreach ( $this->notices as $notice_key => $notice ) {

			?>
			<div class="<?php echo esc_attr( $notice['class'] ); ?>">
				<p>
					<?php echo wp_kses( $notice['message'], array( 'a' => array( 'href' => array() ) ) ); ?>
				</p>
			</div>
			<?php
		}
	}
       
	/**
	 * Gets the main instance.
	 *
	 * Ensures only one instance can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @return \Foxpost_Wc_Shipping_Dependencies
	 */
	public static function instance() {

		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}  
    
} //end class


Foxpost_Wc_Shipping_Dependencies::instance();

/**
 * WooCommerce active detection
 */   
if ( ! function_exists( 'fws_is_woocommerce_active' ) ) {
	function fws_is_woocommerce_active() {
		return Foxpost_Wc_Shipping_Dependencies::woocommerce_active_check();
	}
}
