<?php
  if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly 
/**
 * Register all actions and filters for the plugin
 *
 * @since      1.0.0
 *
 * @package    Foxpost_Wc_Shipping
 * @subpackage Foxpost_Wc_Shipping/includes
 */

require_once dirname(plugin_dir_path( __FILE__ )) . '/abstract-foxpost-wc-shipping-method.php';


/**
 * Class Foxpost_Wc_Shipping_Home_Delivery_Shipping_Method
 */
class Foxpost_Wc_Shipping_Hd_Shipping_Method extends Foxpost_Wc_Shipping_Method {

    /**
     * Constructor for your shipping class
     *
     * @param int $instance_id
     *
     * @since 1.0.0
     */
    public function __construct($instance_id = 0)
    {
        $this->id                 = self::FOXPOST_WC_SHIPPING_HD_SHIPPING_METHOD;
        $this->instance_id        = absint($instance_id);
        $this->method_title       = __('Foxpost - PACKETA Group home delivery shipping', 'foxpost-shipping-method');
        $this->method_description = __('Csomagküldés házhozszállítással', 'foxpost-shipping-method');

        $this->supports = array(
            'shipping-zones',
            'instance-settings',
            'instance-settings-modal',
        );

        $this->init();
    }

    /**
     * Init your settings
     *
     * @since 1.0.0
     *
     * @return void
     */
    public function init()
    {
        $this->init_form_fields();

        $this->enabled             = $this->get_option('enabled');
        $this->title               = $this->get_option('title');
        $this->fee                 = $this->get_option('fee');
        $this->free_fee            = $this->get_option('free_fee');
        $this->max_weight          = $this->get_option('max_weight');
        //$this->phone_required      = $this->get_option('phone_required');
        //$this->seller_own_data     = $this->get_option('seller_own_data');
        $this->default_parcel_size = parent::DEFAULT_PARCEL_SIZE; //$this->get_option('default_parcel_size');

        // Save settings in admin if you have any defined
        add_action(
            'woocommerce_update_options_shipping_' . $this->id, array(
                $this,
                'process_admin_options'
            )
        );
    }

    /**
     * Define settings field for this shipping
     *
     * @since 1.0.0
     *
     * @return void
     */
    function init_form_fields()
    {
        $this->instance_form_fields = array(

            'title'               => array(
                'title'       => __('Method Title', 'foxpost-shipping-method'),
                'type'        => 'text',
                'description' => __('This controls the title which the user sees during checkout.', 'foxpost-shipping-method'),
                'default'     => __('Foxpost home delivery shipping', 'foxpost-shipping-method'),
                'class'       => 'foxpost_wc_shipping_hd_title',
                'desc_tip'    => true,
            ),
            'type'                => array(
                'title'       => __('Fee Type', 'foxpost-shipping-method'),
                'type'        => 'select',
                'description' => __('How to calculate delivery charges', 'foxpost-shipping-method'),
                'default'     => 'fixed',
                'options'     => array(
                    'fixed'   => __('Fixed amount', 'foxpost-shipping-method'),
                    'product' => __('Fixed amount per product', 'foxpost-shipping-method'),
                ),
                'desc_tip'    => true,
            ),
            'fee'                 => array(
                'title'       => __('Delivery Fee', 'foxpost-shipping-method'),
                'type'        => 'price',
                'description' => __('What fee do you want to charge for Foxpost home delivery, disregarded if you choose free. Leave blank to disable.', 'foxpost-shipping-method'),
                'default'     => 1000,
                'placeholder' => wc_format_localized_price(0),
                'class'       => 'foxpost_wc_shipping_hd_fee',
            ),
            'free_fee'            => array(
                /* translators: %s: curreny symbol */
                'title'       => sprintf(esc_html__('Free delivery over x %s', 'foxpost-shipping-method'), self::get_currency_symbol()),
                'type'        => 'price',
                'default'     => 10000000,
                'placeholder' => wc_format_localized_price(0),
                'class'       => 'foxpost_wc_shipping_hd_free_fee',
            ),
            'max_weight'      => array(
                'title'       => __('Max Weight', 'foxpost-shipping-method'),
                'type'        => 'text',
                'description' => __('The maximum weight of a parcel is 25kg.', 'foxpost-shipping-method'),
                'default'     => 25,
            )
        );

    }

    /**
     * This function is used to calculate the shipping cost.
     * Within this function we can check for weights, dimensions and other
     * parameters.
     *
     * @param array $package
     *
     * @since 1.0.0
     *
     * @return void
     */
    public function calculate_shipping($package = array())
    {                                              
        $shipping_total = 0;
        $fee = round($this->fee, 6);
        $free_fee = (int) $this->get_option('free_fee');
        $calculation_type = $this->get_option('type', 'fixed');

        if ($calculation_type === 'fixed') {
            if ($free_fee > 0 && ((int) WC()->cart->get_subtotal() - array_sum(WC()->cart->get_coupon_discount_totals())) >= $free_fee) {
                $shipping_total = 0;
            } else {
                $shipping_total = $fee;
            }
        } elseif ($calculation_type === 'product') {
            foreach (WC()->cart->get_cart() as $item_id => $values) {
                $product = $values['data'];

                if ($values['quantity'] > 0 && $product->needs_shipping()) {
                    $shipping_total += ( $fee * $values['quantity'] );
                }
            }
        }

        $this->add_rate(
            array(
                'label'   => $this->method_title,
                'package' => $package,
                'cost'    => $shipping_total,
            )
        );
    
    }
}
