<?php
/**
 * @link       https://foxpost.hu
 * @since      1.0.0
 *
 * @package    Foxpost_Wc_Shipping
 * @subpackage Foxpost_Wc_Shipping/includes
 * @author     Foxpost.hu <b2chelpdesk@foxpost.hu>
 */

if (!defined('ABSPATH')) {
    die;
}

require_once dirname(dirname(plugin_dir_path(__FILE__)))
             . '/woocommerce/includes/abstracts/abstract-wc-settings-api.php';     
require_once dirname(dirname(plugin_dir_path(__FILE__)))                              
             . '/woocommerce/includes/abstracts/abstract-wc-shipping-method.php';    


/**
 * Class Foxpost_Wc_Shipping_Method
 */
abstract class Foxpost_Wc_Shipping_Method extends WC_Shipping_Method {
    /**                                                
     * Never calculate TAX over Fee
     *
     * @var bool
     *
     * @since 1.0.0
     */
    //public $tax_status = 'no-taxable'; 
    public $tax_status = 'taxable';
    /**                                                
     * Phone required.
     *
     * @var bool
     *
     * @since 1.0.0
     */
    public $phone_required = true;
    /**
     * Default parcel size.
     *
     * @var string
     *
     * @since 1.0.0
     */
    public $default_parcel_size;
    /**
     * Seller remarks.
     *
     * @var string
     *
     * @since 1.0.0
     */
    public $seller_own_data = '';
    /**
     * Maximum weight.
     *
     * @var int
     *
     * @since 1.0.0
     */
    public $max_weight = 0;
    /**
     * Free delivery fee.
     *
     * @var int
     *
     * @since 1.0.0
     */
    public $free_fee = 0;
    /**
     * APT shipping method.
     *
     * @since 1.0.0
     */
    const FOXPOST_WC_SHIPPING_APT_SHIPPING_METHOD
        = 'foxpost_wc_shipping_apt_shipping';
    /**
     * Home delivery shipping merthod.
     *
     * @since 1.0.0
     */
    const FOXPOST_WC_SHIPPING_HD_SHIPPING_METHOD
        = 'foxpost_wc_shipping_hd_shipping';
    /**
     * XS size.
     *
     * @since 1.0.0
     */
    const PARCEL_SIZE_XS = 'XS';
    /**
     * S size.
     *
     * @since 1.0.0
     */
    const PARCEL_SIZE_S = 'S';
    /**
     * M size.
     *
     * @since 1.0.0
     */
    const PARCEL_SIZE_M = 'M';
    /**
     * L size.
     *
     * @since 1.0.0
     */
    const PARCEL_SIZE_L = 'L';
    /**
     * XL size.
     *
     * @since 1.0.0
     */
    const PARCEL_SIZE_XL = 'XL';
    /**
     * S size.
     *
     * @since 1.0.0
     */
    const DEFAULT_PARCEL_SIZE = 'S';

    /**
     * Shipping methods.
     *
     * @var array
     *
     * @since 1.0.0
     */
    protected static $shippingMethods = array();


    /**
     * Available shipping methods.
     *
     * @since 1.0.0
     *
     * @var array
     */
    public static $availableShippingMethods = array();

    /**
     * Is the given shipping name apt shipping method.
     *
     * @param string $method Method to check.
     *
     * @since 1.0.0
     *
     * @return bool
     */
    public static function isMethodAptShippingMethod($method)
    {
        return strpos($method, self::FOXPOST_WC_SHIPPING_APT_SHIPPING_METHOD) === 0;
    }

    /**
     * Is the given shipping name home delivery shipping method.
     *
     * @param string $method Method to check.
     *
     * @since 1.0.0
     *
     * @return bool
     */
    public static function isMethodHdShippingMethod($method)
    {
        return strpos($method, self::FOXPOST_WC_SHIPPING_HD_SHIPPING_METHOD) === 0;
    }

    /**
     * Gets available shipping methods.
     *
     * @since 1.0.0
     *
     * @return array
     */
    public static function getAvailableShippingMethods()
    {
        return array_keys(self::getAvailableShippingMethodsForRegister());
    }

    /**
     * Gets available shipping methods.
     *
     * @since 1.0.0
     *
     * @return array
     */
    public static function getAvailableShippingMethodsForRegister()
    {
        if (empty(self::$availableShippingMethods)) {
            $shipping_method_dir = __DIR__ . '/shipping-methods';        
            foreach (glob($shipping_method_dir . '/*.php') as $file) {
                $file_name                                 = pathinfo($file, PATHINFO_FILENAME);
                include  $shipping_method_dir."/$file_name.php";                  
                //echo     $shipping_method_dir."/$file_name.php";   
                
                $file_name                                 = str_replace('class-', '', $file_name);
                $class_id                                  = str_replace('-', '_', $file_name);
                $class_name                                = self::_mb_convert_case($class_id);
                $class_id                                  = str_replace('_method', '', $class_id);
 
                self::$availableShippingMethods[$class_id] = $class_name;                
            }
        }
        return self::$availableShippingMethods;
    }
    
    /**
     * Whether the given method is a valid method or not.
     *
     * @param string $method Shipping method.
     *
     * @since 1.0.0
     *
     * @return bool
     */
    public static function isValidShippingMethod($method)
    {
        $method = self::normalizeShippingMethodId($method);

        return in_array($method, self::getAvailableShippingMethods(), true);
    }

    /**
     * @param string $chosen_method
     *
     * @return string
     */
    public static function normalizeShippingMethodId($chosen_method)
    {
        return preg_replace('#(^.*?)\:\d+#', "$1", $chosen_method);
    }
    
        
    public static function initShippingMethodInstance($chosen_method, &$shipping_method)
    {
        if (preg_match("/(?<instance_id>\d+)$/", $chosen_method, $matches)) {
            $shipping_method->instance_id = $matches['instance_id'];
            $shipping_method->init();
        }
    }   

    /**
     * @since 1.0.0
     *
     * @return array
     */
    public static function getLabels()
    {
        static $labels;
        if (null === $labels) {
            $labels = array(
                self::FOXPOST_WC_SHIPPING_APT_SHIPPING_METHOD           => __('Foxpost - PACKETA Group parcel locker shipping', 'foxpost-shipping-method'),
                self::FOXPOST_WC_SHIPPING_HD_SHIPPING_METHOD            => __('Foxpost - PACKETA Group home delivery shipping', 'foxpost-shipping-method'),
            );
        }

        return $labels;
    }

    /**
     * Gets label.
     *
     * @param string $method  Shipping method.
     * @param string $default Default value.
     *
     * @since 1.0.0
     *
     * @return mixed|string
     */
    public static function getLabel($method, $default = '')
    {
        $labels = self::getLabels();

        return isset($labels[$method]) ? $labels[$method] : $default;
    }


    /**
     * Check if shipping is available
     *
     * @param array $package
     *
     * @since 1.0.0
     *
     * @return bool
     */
    public function is_available($package)
    {
        return !( 'no' === $this->enabled );
    }

    /**
     * Gets current currency.
     *
     * @since 1.0.0
     *
     * @return string
     */
    public static function get_currency()
    {
        return get_woocommerce_currency();
    }

    /**
     * Gets currency symbol.
     *
     * @since 1.0.0
     *
     * @return string
     */
    public static function get_currency_symbol()
    {
        return get_woocommerce_currency_symbol();
    }



    /**
     * @since 1.0.3
     *
     * @return array|\WP_Error
     */
    public static function getAptDataFromApi()
    {
        $jsonUrl = FoxPostApi::APMJSON;
        return wp_remote_get($jsonUrl);
    }

    /**
     * @since 1.0.0
     *
     * @return array
     */
    public static function getParcelSizes()
    {
        return array(
            self::PARCEL_SIZE_XS => __('XS', 'foxpost-shipping-method'),
            self::PARCEL_SIZE_S  => __('S', 'foxpost-shipping-method'),
            self::PARCEL_SIZE_M  => __('M', 'foxpost-shipping-method'),
            self::PARCEL_SIZE_L  => __('L', 'foxpost-shipping-method'),
            self::PARCEL_SIZE_XL => __('XL', 'foxpost-shipping-method'),
        );
    }

    /**
     * @since 1.0.0
     *
     * @return string
     */
    public static function getDefaultParcelSize()
    {
        return self::DEFAULT_PARCEL_SIZE;
    }


    /**
     * @since 1.0.0
     *
     * @return array
     */
    public static function getLabelSizes()  
    {
        return array(
            'A5' => __('A5', 'foxpost-shipping-method'),
            'A6' => __('A6', 'foxpost-shipping-method'),
            'A7'  => __('A7', 'foxpost-shipping-method'),
            '_85X85' => __('85X85mmm', 'foxpost-shipping-method')
        );
    }
    /**
     * @since 1.0.0
     *
     * @return string
     */
    public static function getDefaultLabelSize()
    {
        return 'A7';
    }    
    
 
    
    /**
     * Convert a UTF-8 encoded string to title/proper case
     *    (uppercase every first character in each word, lower case all other characters)
     *
     * @param string $string UTF-8 encoded string
     *
     * @return string
     */    
    public static function _mb_convert_case($string = '')
    {
        if (function_exists('mb_convert_case')) {
            return mb_convert_case($string, MB_CASE_TITLE, 'UTF-8');
        }

        return ucwords($string);
    }  
    
    
}
